/*
*  Object %name    : %
*  State           :  %state%
*  Creation date   :  Wen Mar 20 10:17:10 2007
*  Last modified   :  %modify_time%
*/
/** @file 
*  \brief A brief description of this module 
*
*  \version CE2_TST_DH_KG.c#1:csrc:1
*  \author Aliaksandr Masiuk
*  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
*           All Rights reserved
*/

/************* Include Files ****************/

#include <memory.h>
#include "CryptoEngine2.h"
#include "CE2_TST_DH_KG.h"
#include "CE2_TST_Common.h"
#include "CE2_DH_KG.h"

/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

TST_DH_TestDataStruct TST_DH_KG_Acceptance_TestDataStruct[] = 
{
	{
		/* ANSI X9.42-2001 [Annex D.1.1] */
		{ "Acceptance test - DH P-1024/Q-160/Seed-20/ANSI X9.42-2001 [Annex D.1.1]" },
		/* L in bits (size of P) */
		1024,
		/* m in bits (size of Q) */
		160,
		/* seed size in bits */
		20 * 8,
		/* P */
		{
			0xE0, 0x01, 0xE8, 0x96, 0x7D, 0xB4, 0x93, 0x53, 0xE1, 0x6F, 0x8E, 0x89, 0x22, 0x0C, 0xCE, 0xFC,
			0x5C, 0x5F, 0x12, 0xE3, 0xDF, 0xF8, 0xF1, 0xD1, 0x49, 0x90, 0x12, 0xE6, 0xEF, 0x53, 0xE3, 0x1F,
			0x02, 0xEA, 0xCC, 0x5A, 0xDD, 0xF3, 0x37, 0x89, 0x35, 0xC9, 0x5B, 0x21, 0xEA, 0x3D, 0x6F, 0x1C,
			0xD7, 0xCE, 0x63, 0x75, 0x52, 0xEC, 0x38, 0x6C, 0x0E, 0x34, 0xF7, 0x36, 0xAD, 0x95, 0x17, 0xEF,
			0xFE, 0x5E, 0x4D, 0xA7, 0xA8, 0x6A, 0xF9, 0x0E, 0x2C, 0x22, 0x8F, 0xE4, 0xB9, 0xE6, 0xD8, 0xF8,
			0xF0, 0x2D, 0x20, 0xAF, 0x78, 0xAB, 0xB6, 0x92, 0xAC, 0xBC, 0x4B, 0x23, 0xFA, 0xF2, 0xC5, 0xCC,
			0xD4, 0x9A, 0x0C, 0x9A, 0x8B, 0xCD, 0x91, 0xAC, 0x0C, 0x55, 0x92, 0x01, 0xE6, 0xC2, 0xFD, 0x1F,
			0x47, 0xC2, 0xCB, 0x2A, 0x88, 0xA8, 0x3C, 0x21, 0x0F, 0xC0, 0x54, 0xDB, 0x29, 0x2D, 0xBC, 0x45
		},
		/* Q */
		{
			0x86, 0x47, 0x17, 0xA3, 0x9E, 0x6A, 0xEA, 0x7E, 0x87, 0xC4, 0x32, 0xEE, 0x77, 0x43, 0x15, 0x16,
			0x96, 0x70, 0xC4, 0x99
		},
		/* G */
		{
			0x1C, 0xE0, 0xF6, 0x69, 0x26, 0x46, 0x11, 0x97, 0xEF, 0x45, 0xC4, 0x65, 0x8B, 0x83, 0xB8, 0xAB,
			0x04, 0xA9, 0x22, 0x42, 0x68, 0x50, 0x4D, 0x05, 0xB8, 0x19, 0x83, 0x99, 0xDD, 0x71, 0x37, 0x18,
			0xCC, 0x1F, 0x24, 0x5D, 0x47, 0x6C, 0xCF, 0x61, 0xA2, 0xF9, 0x34, 0x93, 0xF4, 0x1F, 0x55, 0x52,
			0x48, 0x65, 0x57, 0xE6, 0xD4, 0xCA, 0xA8, 0x00, 0xD6, 0xD0, 0xDB, 0x3C, 0xBF, 0x5A, 0x95, 0x4B,
			0x20, 0x8A, 0x4E, 0xBA, 0xF7, 0xE6, 0x49, 0xFB, 0x61, 0x24, 0xD8, 0xA2, 0x1E, 0xF2, 0xF2, 0x2B,
			0xAA, 0xAE, 0x29, 0x21, 0x10, 0x19, 0x10, 0x51, 0x46, 0x47, 0x31, 0xB6, 0xCC, 0x3C, 0x93, 0xDC,
			0x6E, 0x80, 0xBA, 0x16, 0x0B, 0x66, 0x64, 0xA5, 0x6C, 0xFA, 0x96, 0xEA, 0xF1, 0xB2, 0x83, 0x39,
			0x8E, 0xB4, 0x61, 0x64, 0xE5, 0xE9, 0x43, 0x84, 0xEE, 0x02, 0x24, 0xE7, 0x1F, 0x03, 0x7C, 0x23
		},
		/* G size in bytes */
		128,
		/* J */
		{
			0
		},
		/* J size in bytes */
		0,
		/* seed S */
		{
			0xD5, 0x01, 0x4E, 0x4B, 0x60, 0xEF, 0x2B, 0xA8, 0xB6, 0x21, 0x1B, 0x40, 0x62, 0xBA, 0x32, 0x24,
			0xE0, 0x42, 0x7F, 0x4F
		},
		/* generate seed */
		0,
		/* pgenCounter */
		246,
		/* Public keys for U and V, ANSI X9.42-2001 [Annex D.2.2] */
		/* yU */
		{
			0xBB, 0xE9, 0x18, 0xDD, 0x4B, 0x2B, 0x94, 0x1B, 0x10, 0x0E, 0x88, 0x35, 0x28, 0x68, 0xFC, 0x62,
			0x04, 0x38, 0xA6, 0xDB, 0x32, 0xA6, 0x9E, 0xEE, 0x6C, 0x6F, 0x45, 0x1C, 0xA3, 0xA6, 0xD5, 0x37,
			0x77, 0x75, 0x5B, 0xC1, 0x37, 0x0A, 0xCE, 0xFE, 0x2B, 0x8F, 0x13, 0xA9, 0x14, 0x2C, 0x5B, 0x44,
			0x15, 0x78, 0x86, 0x30, 0xD6, 0x95, 0xB1, 0x92, 0x20, 0x63, 0xA3, 0xCF, 0x9D, 0xEF, 0x65, 0x61,
			0x27, 0x4D, 0x24, 0x01, 0xE7, 0xA1, 0x45, 0xF2, 0xD8, 0xB9, 0x3A, 0x45, 0x17, 0xF4, 0x19, 0xD0,
			0x5E, 0xF8, 0xCB, 0x35, 0x59, 0x37, 0x9D, 0x04, 0x20, 0xA3, 0xBF, 0x02, 0xAD, 0xFE, 0xA8, 0x60,
			0xB2, 0xC3, 0xEE, 0x85, 0x58, 0x90, 0xF3, 0xB5, 0x57, 0x2B, 0xB4, 0xEF, 0xD7, 0x8F, 0x37, 0x68,
			0x78, 0x7C, 0x71, 0x52, 0x9D, 0x5E, 0x0A, 0x61, 0x4F, 0x09, 0x89, 0x92, 0x39, 0xF7, 0x4B, 0x01
		},
		/* yU size in bytes */
		128,
		/* yV */
		{
			0xA3, 0xF5, 0x7D, 0xBE, 0x9E, 0x2F, 0x0A, 0xDA, 0xA9, 0x4E, 0x4E, 0x6A, 0xF0, 0xE0, 0x71, 0x47,
			0x0E, 0x2E, 0x41, 0x2E, 0xDE, 0x73, 0x2A, 0x62, 0x14, 0xC3, 0x7C, 0x26, 0xD4, 0xE9, 0x9A, 0x54,
			0xBA, 0x3D, 0xE7, 0x49, 0x85, 0x95, 0x0E, 0xE9, 0x14, 0xB2, 0x90, 0x22, 0x91, 0xDC, 0xFF, 0x61,
			0xB2, 0xFC, 0xD1, 0xD0, 0x1B, 0x11, 0x14, 0xB6, 0x02, 0x64, 0x2B, 0x26, 0x5D, 0x88, 0xEA, 0x8D,
			0xBB, 0xE2, 0x07, 0x0B, 0x48, 0xFB, 0x01, 0x53, 0x55, 0x1E, 0x59, 0x51, 0x36, 0xF2, 0xF9, 0xD1,
			0x97, 0xFB, 0x66, 0x12, 0x84, 0x5D, 0xED, 0xB8, 0x9B, 0x2D, 0x3E, 0x2B, 0x8C, 0xEB, 0x2A, 0x72,
			0x40, 0x9D, 0x55, 0x4C, 0xED, 0xEB, 0x55, 0x02, 0xFF, 0x8C, 0xB0, 0x2E, 0x03, 0x65, 0x3F, 0x41,
			0xB1, 0xAC, 0xA3, 0x30, 0x6B, 0xFF, 0x6D, 0xF4, 0x6D, 0xE6, 0xE1, 0x0F, 0x86, 0x7C, 0x43, 0x64
		},
		/* yV size in bytes */
		128
	},
	{
		/* ANSI X9.42-2001 [Annex D.1.2] */
		{ "Acceptance test - DH P-1024/Q-160/Seed-20/ANSI X9.42-2001 [Annex D.1.2]" },
		/* L in bits (size of P) */
		1024,
		/* m in bits (size of Q) */
		160,
		/* seed size in bits */
		20 * 8,
		/* P */
		{
			0xD7, 0x57, 0x26, 0x2C, 0x45, 0x84, 0xC4, 0x4C, 0x21, 0x1F, 0x18, 0xBD, 0x96, 0xE5, 0xF0, 0x61,
			0xC4, 0xF0, 0xA4, 0x23, 0xF7, 0xFE, 0x6B, 0x6B, 0x85, 0xB3, 0x4C, 0xEF, 0x72, 0xCE, 0x14, 0xA0,
			0xD3, 0xA5, 0x22, 0x2F, 0xE0, 0x8C, 0xEC, 0xE6, 0x5B, 0xE6, 0xC2, 0x65, 0x85, 0x48, 0x89, 0xDC,
			0x1E, 0xDB, 0xD1, 0x3E, 0xC8, 0xB2, 0x74, 0xDA, 0x9F, 0x75, 0xBA, 0x26, 0xCC, 0xB9, 0x87, 0x72,
			0x36, 0x02, 0x78, 0x7E, 0x92, 0x2B, 0xA8, 0x44, 0x21, 0xF2, 0x2C, 0x3C, 0x89, 0xCB, 0x9B, 0x06,
			0xFD, 0x60, 0xFE, 0x01, 0x94, 0x1D, 0xDD, 0x77, 0xFE, 0x6B, 0x12, 0x89, 0x3D, 0xA7, 0x6E, 0xEB,
			0xC1, 0xD1, 0x28, 0xD9, 0x7F, 0x06, 0x78, 0xD7, 0x72, 0x2B, 0x53, 0x41, 0xC8, 0x50, 0x6F, 0x35,
			0x82, 0x14, 0xB1, 0x6A, 0x2F, 0xAC, 0x4B, 0x36, 0x89, 0x50, 0x38, 0x78, 0x11, 0xC7, 0xDA, 0x33

		},
		/* Q */
		{
			0xC7, 0x73, 0x21, 0x8C, 0x73, 0x7E, 0xC8, 0xEE, 0x99, 0x3B, 0x4F, 0x2D, 0xED, 0x30, 0xF4, 0x8E,
			0xDA, 0xCE, 0x91, 0x5F
		},
		/* G */
		{
			0x82, 0x26, 0x90, 0x09, 0xE1, 0x4E, 0xC4, 0x74, 0xBA, 0xF2, 0x93, 0x2E, 0x69, 0xD3, 0xB1, 0xF1,
			0x85, 0x17, 0xAD, 0x95, 0x94, 0x18, 0x4C, 0xCD, 0xFC, 0xEA, 0xE9, 0x6E, 0xC4, 0xD5, 0xEF, 0x93,
			0x13, 0x3E, 0x84, 0xB4, 0x70, 0x93, 0xC5, 0x2B, 0x20, 0xCD, 0x35, 0xD0, 0x24, 0x92, 0xB3, 0x95,
			0x9E, 0xC6, 0x49, 0x96, 0x25, 0xBC, 0x4F, 0xA5, 0x08, 0x2E, 0x22, 0xC5, 0xB3, 0x74, 0xE1, 0x6D,
			0xD0, 0x01, 0x32, 0xCE, 0x71, 0xB0, 0x20, 0x21, 0x70, 0x91, 0xAC, 0x71, 0x7B, 0x61, 0x23, 0x91,
			0xC7, 0x6C, 0x1F, 0xB2, 0xE8, 0x83, 0x17, 0xC1, 0xBD, 0x81, 0x71, 0xD4, 0x1E, 0xCB, 0x83, 0xE2,
			0x10, 0xC0, 0x3C, 0xC9, 0xB3, 0x2E, 0x81, 0x05, 0x61, 0xC2, 0x16, 0x21, 0xC7, 0x3D, 0x6D, 0xAA,
			0xC0, 0x28, 0xF4, 0xB1, 0x58, 0x5D, 0xA7, 0xF4, 0x25, 0x19, 0x71, 0x8C, 0xC9, 0xB0, 0x9E, 0xEF

		},
		/* G size in bytes */
		128,
		/* J */
		{
			0
		},
		/* J size in bytes */
		0,
		/* seed S */
		{
			0xD5, 0x01, 0x4E, 0x4B, 0x60, 0xEF, 0x2B, 0xA8, 0xB6, 0x21, 0x1B, 0x40, 0x62, 0xBA, 0x32, 0x24,
			0xE0, 0x42, 0x7D, 0xD3
		},
		/* generate seed */
		0,
		/* pgenCounter */
		371,
		/* Public keys for U and V, ANSI X9.42-2001 [Annex D.2.2] */
		/* yU */
		{ 0x00 },
		/* yU size in bytes */
		0,
		/* yV */
		{ 0x00 },
		/* yV size in bytes */
		0
	}
};

const int TST_NumberOf_DH_KG_AcceptanceTests = sizeof(TST_DH_KG_Acceptance_TestDataStruct)/sizeof(TST_DH_KG_Acceptance_TestDataStruct[0]); 

/************* Private function prototype ***********************/
/************************ Public Functions ******************************/

/*****************************************************************************
* Function Name:                                                           
* 	CE2_ATP_TST_DH_CreateDomainParams_Acceptance
* 
* Inputs:
* 	out - pointer to output log file;
*
* Outputs:
* 	CE2Error_t - Function error return		    					  		
*
* Description:																	
* 	This function Is a Acceptance test for the Diffie-Hellman key agreement 
*  schemes according to ANSI9.42 using full keys test method: P, Q, G, Seed.
*
* Algorithm:		
* 1. Create P, Q, G with using CE2_DH_CreateDomainParams method with test values;
* 2. Validate output data: P, Q, G.
******************************************************************************/
CE2Error_t CE2_ATP_TST_DH_CreateDomainParams_Acceptance(FILE* out)
{
	CE2Error_t	TST_TestStatus = 0,
				error;
	DxUint32_t	TST_Index;

	static TST_DH_TestDataStruct data_out = { 0 };

	PrintTestBeginMsg(out, __FUNCTION__);

	/* Test for all available test vectors */
	for (TST_Index = 0; TST_Index < (DxUint32_t)TST_NumberOf_DH_KG_AcceptanceTests; ++TST_Index)
	{
		/* Initialize buffer sizes */
		data_out.modPsizeBits = TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modPsizeBits;
		data_out.orderQsizeBits = TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQsizeBits;
		data_out.JsizeBytes_ptr = TST_MAX_DH_KEY_SIZE;
		data_out.generGsizeBytes_ptr = TST_DH_KG_Acceptance_TestDataStruct[TST_Index].generGsizeBytes_ptr;
		data_out.seedSizeBits = TST_DH_KG_Acceptance_TestDataStruct[TST_Index].seedSizeBits;
		data_out.generateSeed = TST_DH_KG_Acceptance_TestDataStruct[TST_Index].generateSeed;

		memcpy(
			data_out.seedS_ptr,
			TST_DH_KG_Acceptance_TestDataStruct[TST_Index].seedS_ptr,
			(data_out.seedSizeBits >> 3) + ((data_out.seedSizeBits & 0x07) ? (1) : (0)));

		/* Create params using CE2_DH_CreateDomainParams */
		error =
			CE2_DH_CreateDomainParams(
				data_out.modPsizeBits,
				data_out.orderQsizeBits,
				data_out.seedSizeBits,
				data_out.modP_ptr,
				data_out.orderQ_ptr,
				data_out.generatorG_ptr,
				&data_out.generGsizeBytes_ptr,
				data_out.factorJ_ptr,
				&data_out.JsizeBytes_ptr,
				data_out.seedS_ptr,
				data_out.generateSeed,
				&data_out.pgenCounter_ptr);

		/* Validate create params */
		if (error != CE2_OK)
		{
			fprintf(out, "\nCan't create domain params.");
			goto local_error;
		}

		/* Validate created params */

		/* 1. Check created P, Q, pgenCounter to be equals to test */

		/* Validate P */
		error =
			memcmp(
				data_out.modP_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modP_ptr,
				data_out.modPsizeBits >> 3);

		if (error != 0)
		{
			fprintf(out, "\nCheck created domain params failed: P.");
			goto local_error;
		}

		/* Validate Q */
		error =
			memcmp(
				data_out.orderQ_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQ_ptr,
				data_out.orderQsizeBits >> 3);

		if (error != 0)
		{
			fprintf(out, "\nCheck created domain params failed: Q.");
			goto local_error;
		}

		/* Validate pgenCounter */
		if (data_out.pgenCounter_ptr != TST_DH_KG_Acceptance_TestDataStruct[TST_Index].pgenCounter_ptr)
		{
			fprintf(out, "\nCheck created domain params failed: pgenCounter.");
			goto local_error;
		}

		/* 2. Check G and other created params using CE2_DH_CheckDomainParams */
		error =
			CE2_DH_CheckDomainParams(
				data_out.modP_ptr,
				data_out.modPsizeBits >> 3,
				data_out.orderQ_ptr,
				data_out.orderQsizeBits >> 3,
				data_out.generatorG_ptr,
				data_out.generGsizeBytes_ptr,
				data_out.seedS_ptr,
				data_out.seedSizeBits,
				data_out.pgenCounter_ptr);

		/* Validate check create params */
		if (error != CE2_OK)
		{
			fprintf(out, "\nCheck created domain params failed.");
			goto local_error;
		}

		continue;
local_error:
		fprintf(
			out,
			"\n %s[%d test %s]",
			__FUNCTION__,
			TST_Index,
			TST_DH_KG_Acceptance_TestDataStruct[TST_Index].TST_Name);

		TST_TestStatus |= error;
	}

	PrintTestEndMsg(out, __FUNCTION__, TST_TestStatus);

	return TST_TestStatus;
} /* End of CE2_ATP_TST_DH_CreateDomainParams_Acceptance */

/*****************************************************************************
* Function Name:                                                           
* 	CE2_ATP_TST_DH_CreateDomainParams_Acceptance
* 
* Inputs:
* 	out - pointer to output log file;
*
* Outputs:
* 	CE2Error_t - Function error return		    					  		
*
* Description:																	
* 	This function Is a test for the Diffie-Hellman key agreement 
*  schemes according to ANSI9.42 using full keys test method: P, Q, G, Seed
*
* Algorithm:		
* 1. Create P, Q, G with using CE2_DH_CreateDomainParams method;
* 2. Validate output data: P, Q, Seed, G with using CE2_DH_CheckDomainParams.
******************************************************************************/
CE2Error_t CE2_ATP_TST_DH_CreateDomainParams(FILE* out)
{
	CE2Error_t	TST_TestStatus = 0,
				error;
	DxUint32_t	TST_SeedMaxSize,
				TST_SeedStep,
				TST_QMaxSize,
				TST_QStep;

	static TST_DH_TestDataStruct data_out = { 0 };

	PrintTestBeginMsg(out, __FUNCTION__);

	data_out.generateSeed = 1;

	/* Test for all available P sizes */
	for (
		data_out.modPsizeBits = 1024;
		data_out.modPsizeBits < CE2_DH_MAX_MOD_SIZE_IN_BYTES << 2;
		data_out.modPsizeBits += 256)
	{
		/* Initialize seed test values */
		TST_SeedMaxSize = data_out.modPsizeBits >> 1;
		TST_SeedStep = (TST_SeedMaxSize - 20 * 8) /*>> 1*/;

		TST_QMaxSize = data_out.modPsizeBits;
		TST_QStep = ((TST_QMaxSize - CE2_DH_SEED_MIN_SIZE_IN_BITS) /*>> 1*/) >> 5;

		if (TST_QStep <= 0)
			TST_QStep = 1;

		TST_QStep <<= 5;

		/* Test for all available Q sizes */
		for (
			data_out.orderQsizeBits = 160;
			data_out.orderQsizeBits <= TST_QMaxSize;
			data_out.orderQsizeBits += TST_QStep)
		{
			/* Test for all available seed sizes */
			for (
				data_out.seedSizeBits = data_out.orderQsizeBits;
				data_out.seedSizeBits <= TST_SeedMaxSize;
				data_out.seedSizeBits += TST_SeedStep)
			{
				if (data_out.seedSizeBits < CE2_DH_SEED_MIN_SIZE_IN_BITS)
					continue;

				/* Initialize buffer sizes */
				data_out.JsizeBytes_ptr = TST_MAX_DH_KEY_SIZE;
				data_out.generGsizeBytes_ptr = data_out.modPsizeBits >> 3;

				/* Create params using CE2_DH_CreateDomainParams */
				error =
					CE2_DH_CreateDomainParams(
						data_out.modPsizeBits,
						data_out.orderQsizeBits,
						data_out.seedSizeBits,
						data_out.modP_ptr,
						data_out.orderQ_ptr,
						data_out.generatorG_ptr,
						&data_out.generGsizeBytes_ptr,
						data_out.factorJ_ptr,
						&data_out.JsizeBytes_ptr,
						data_out.seedS_ptr,
						data_out.generateSeed,
						&data_out.pgenCounter_ptr);

				/* Validate create params */
				if (error != CE2_OK)
				{
					fprintf(out, "Can't create domain params.");
					goto local_error;
				}

				/* Check created params using CE2_DH_CheckDomainParams */
				error =
					CE2_DH_CheckDomainParams(
						data_out.modP_ptr,
						data_out.modPsizeBits >> 3,
						data_out.orderQ_ptr,
						data_out.orderQsizeBits >> 3,
						data_out.generatorG_ptr,
						data_out.generGsizeBytes_ptr,
						data_out.seedS_ptr,
						data_out.seedSizeBits,
						data_out.pgenCounter_ptr);

				/* Validate check create params */
				if (error != CE2_OK)
				{
					fprintf(out, "Created domain params aren't valid.");
					goto local_error;
				}

				continue;
local_error:
				fprintf(
					out,
					" %s[test]:P(%d bits), Q(%d bits), Seed(%d bits)\n",
					__FUNCTION__,
					data_out.modPsizeBits,
					data_out.orderQsizeBits,
					data_out.seedSizeBits);

				TST_TestStatus |= error;
			}
		}
	}

	PrintTestEndMsg(out, __FUNCTION__, TST_TestStatus);

	return TST_TestStatus;
} /* End of CE2_ATP_TST_DH_CreateDomainParams */

/*****************************************************************************
* Function Name:                                                           
* 	CE2_ATP_TST_DH_CheckDomainParams_Acceptance
* 
* Inputs:
* 	out - pointer to output log file;
*
* Outputs:
* 	CE2Error_t - Function error return		    					  		
*
* Description:																	
* 	This function Is a Acceptance test for the Diffie-Hellman key agreement 
*  schemes according to ANSI9.42 using full keys test method: P, Q, G, Seed.
*
* Algorithm:		
* 1. Check P, Q, G and Seed using CE2_DH_CheckDomainParams method.
******************************************************************************/
CE2Error_t CE2_ATP_TST_DH_CheckDomainParams_Acceptance(FILE* out)
{
	CE2Error_t	TST_TestStatus = 0,
				error;
	DxUint32_t	TST_Index;

	PrintTestBeginMsg(out, __FUNCTION__);

	/* Test for all available test vectors */
	for (TST_Index = 0; TST_Index < (DxUint32_t)TST_NumberOf_DH_KG_AcceptanceTests; ++TST_Index)
	{
		/* Check created params using CE2_DH_CheckDomainParams */
		error =
			CE2_DH_CheckDomainParams(
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modP_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modPsizeBits >> 3,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQ_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQsizeBits >> 3,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].generatorG_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].generGsizeBytes_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].seedS_ptr,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].seedSizeBits,
				TST_DH_KG_Acceptance_TestDataStruct[TST_Index].pgenCounter_ptr);

		/* Validate check create params */
		if (error != CE2_OK)
		{
			fprintf(out, "\nCheck params failed.");
			goto local_error;
		}

		continue;
local_error:
		fprintf(
			out,
			"\n %s[%d test %s]",
			__FUNCTION__,
			TST_Index,
			TST_DH_KG_Acceptance_TestDataStruct[TST_Index].TST_Name);

		TST_TestStatus |= error;
	}

	PrintTestEndMsg(out, __FUNCTION__, TST_TestStatus);

	return TST_TestStatus;
} /* End of CE2_ATP_TST_DH_CheckDomainParams_Acceptance */

/*****************************************************************************
* Function Name:                                                           
* 	CE2_ATP_TST_DH_CheckDomainParams_Acceptance
* 
* Inputs:
* 	out - pointer to output log file;
*
* Outputs:
* 	CE2Error_t - Function error return		    					  		
*
* Description:																	
* 	This function Is a Acceptance test for the Diffie-Hellman key agreement 
*  schemes according to ANSI9.42 using full keys test method: yU, yV.
*
* Algorithm:		
* 1. Check public keys using CE2_DH_CheckPubKey method with test values.
******************************************************************************/
CE2Error_t CE2_ATP_TST_DH_CheckPubKey_Acceptance(FILE* out)
{
	CE2Error_t	TST_TestStatus = 0,
				error;
	DxUint32_t	TST_Index;

	PrintTestBeginMsg(out, __FUNCTION__);

	/* Test for all available test vectors */
	for (TST_Index = 0; TST_Index < (DxUint32_t)TST_NumberOf_DH_KG_AcceptanceTests; ++TST_Index)
	{
		/* Check created params using CE2_DH_CheckDomainParams */

		/* Check yU public key value if available */
		if (TST_DH_KG_Acceptance_TestDataStruct[TST_Index].yUSizeBytes > 0)
		{
			error =
				CE2_DH_CheckPubKey(
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modP_ptr,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modPsizeBits >> 3,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQ_ptr,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQsizeBits >> 3,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].yU,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].yUSizeBytes);

			/* Validate check public key */
			if (error != CE2_OK)
			{
				fprintf(out, "\nCheck public key failed.");
				goto local_error;
			}
		}


		/* Check yV public key value if available */
		if (TST_DH_KG_Acceptance_TestDataStruct[TST_Index].yVSizeBytes > 0)
		{
			error =
				CE2_DH_CheckPubKey(
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modP_ptr,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].modPsizeBits >> 3,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQ_ptr,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].orderQsizeBits >> 3,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].yV,
					TST_DH_KG_Acceptance_TestDataStruct[TST_Index].yVSizeBytes);

			/* Validate check public key */
			if (error != CE2_OK)
			{
				fprintf(out, "\nCheck public key failed.");
				goto local_error;
			}
		}

		continue;
local_error:
		fprintf(
			out,
			"\n %s[%d test %s]",
			__FUNCTION__,
			TST_Index,
			TST_DH_KG_Acceptance_TestDataStruct[TST_Index].TST_Name);

		TST_TestStatus |= error;
	}

	PrintTestEndMsg(out, __FUNCTION__, TST_TestStatus);

	return TST_TestStatus;
} /* End of CE2_ATP_TST_DH_CheckPubKey_Acceptance */
